/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;

using Borland.Eco.DataRepresentation;
using Borland.Eco.Interfaces;
using Borland.Eco.Internal.DefaultImpl;
using Borland.Eco.Services;

namespace Borland.Eco.Persistence
{
	[Serializable]
	public abstract class UpdatePrecondition { }
	[Serializable]
	public abstract class UpdateResult
	{
		public abstract bool Failed { get; }
		public abstract string FailureReason { get; }
	}
	
	[Serializable]
	public sealed class OkResult: UpdateResult
	{
		public override bool Failed
		{ 
		  get { return false; } 
		}
		public override string FailureReason
		{
			get { return string.Empty; }
		}
	}
	
	[Serializable]
	public sealed class OptimisticLockingPrecondition: UpdatePrecondition
	{
		private Datablock m_Datablock = new Datablock();

		public Datablock Datablock
		{
			get { return m_Datablock; }
		}
		public bool HasOptimisticLocks
		{
			get { return Datablock.Count > 0; }
		}
	}
  [Serializable]
	public sealed class OptimisticLockingResult: UpdateResult
	{
		private readonly Datablock m_Conflicts = new Datablock();

		public Datablock Conflicts
		{
			get { return m_Conflicts; }
		}

		public override bool Failed
		{
			get { return Conflicts.Count > 0; }
		}
		
		public override string FailureReason
		{
			get { return InterfacesStringRes.sOptimisticLockingFailed(Conflicts.Count); }
		}
	}
	[Serializable]
	public abstract class VersionConst
	{
		public const int NonVersioned = -1;
		public const int CurrentVersion = int.MaxValue; 
	}

	[Serializable]
	public abstract class AbstractCondition
	{
		private readonly int m_ClassId;
		private readonly int m_Version = VersionConst.CurrentVersion;
		public int ClassId
		{
			get { return m_ClassId; }
		}
		public int Version
		{
			get { return m_Version; }
		}
		protected AbstractCondition(int classId, int version)
		{
			m_ClassId = classId;
			m_Version = version;
		}

		protected AbstractCondition(int classId)
		{
			m_ClassId = classId;
			m_Version = VersionConst.CurrentVersion;
		}

	}

	[Serializable]
	public sealed class AllInstancesCondition: AbstractCondition
	{
		public AllInstancesCondition(int classId, int version): base(classId, version) {}
		public AllInstancesCondition(int classId): base(classId) {}
	}

	[Serializable]
	public sealed class ChangePointCondition: AbstractCondition
	{
		private readonly ObjectIdList m_ObjectIds;
		private readonly int m_StartVersion;
		private readonly int m_EndVersion;
		private readonly int[] m_MemberIds;

		public ChangePointCondition(int classId, ObjectIdList objectIds, int startVersion,
			int endVersion, int[] memberIds): base(classId)
		{
			m_ObjectIds = objectIds;
			m_StartVersion = startVersion;
			m_EndVersion = endVersion;
			m_MemberIds = memberIds;
		}

		public ObjectIdList ObjectIds
		{
			get { return m_ObjectIds; }
		}
		public int StartVersion
		{
			get { return m_StartVersion; }
		}
		public int EndVersion
		{
			get { return m_EndVersion; }
		}

		public int[] MemberIds
		{
			get { return m_MemberIds; }
		}
	}

	[Serializable]
	public sealed class DBChange
	{
		public enum ChangeKind {None, EmbeddedMembersChanged, MembersChanged, ObjectCreated, ObjectDeleted}

		private ChangeKind m_Kind = ChangeKind.None;
		private readonly ObjectId m_ObjectId;
		private int[] m_MemberIds;

		public  DBChange (ObjectId objectId)
		{
			m_ObjectId = objectId;
		}

		public bool EmbeddedMembersChanged
		{
			get
			{
				return ((m_Kind == ChangeKind.EmbeddedMembersChanged) ||
				(((m_Kind == ChangeKind.MembersChanged) || (m_Kind == ChangeKind.ObjectCreated))
				 && (Array.IndexOf(m_MemberIds, -1, m_MemberIds.GetLowerBound(0), m_MemberIds.Length - 1) != -1)));
			}
			set
			{
				if (m_Kind == ChangeKind.None)
					m_Kind = ChangeKind.EmbeddedMembersChanged;
				else if ((m_Kind == ChangeKind.MembersChanged) || (m_Kind == ChangeKind.ObjectCreated))
					MemberChanged(-1); // Add explicit -1
			}
		}

		public void MemberChanged(int memberId)
		{
			if (m_Kind == ChangeKind.None)
			{
				m_Kind =  ChangeKind.MembersChanged;
				m_MemberIds = new int[0];
			}
			else if (m_Kind == ChangeKind.EmbeddedMembersChanged)
			{
				m_MemberIds = new int[1];
				m_MemberIds[0] = -1;
				m_Kind =  ChangeKind.MembersChanged;
			}
			if (m_Kind == ChangeKind.ObjectDeleted)
				return; // Don't keep track of changes in deleted objects
			if (m_MemberIds == null)
				m_MemberIds = new int[0]; // for ChangeKind.ObjectCreated
			if (Array.IndexOf(m_MemberIds, memberId, m_MemberIds.GetLowerBound(0), m_MemberIds.Length) == -1)
			{
				int[] oldIds = m_MemberIds;
				m_MemberIds =  new int[oldIds.Length + 1];
				oldIds.CopyTo(m_MemberIds, 0);
				m_MemberIds[m_MemberIds.Length - 1] = memberId;
			}
		}

		public void ObjectDeleted()
		{
			m_Kind =  ChangeKind.ObjectDeleted;
			m_MemberIds = null;
		}

		public void ObjectCreated()
		{
			m_Kind = ChangeKind.ObjectCreated;
		}

		public ChangeKind Kind
		{
			get { return m_Kind; }
		}

		public int[] MemberIds
		{
			get { return m_MemberIds; }
		}

		public ObjectId ObjectId
		{
			get { return m_ObjectId; }
		}
	}
	
	///<summary>
	///Collection of changes
	///</summary>
	[Serializable]
	public sealed class DBChangeCollection: IEnumerable
	{
		private ArrayList m_List; // Only one of list and Hashtable exist
		private Hashtable m_Hashtable;
		private bool m_All;

		public void Strip() // Make faster to serialize
		{
			m_List = new ArrayList(m_Hashtable.Values);
			m_Hashtable = null;
		}

		private Hashtable EnsuredHashTable
		{
			get
			{
				if (m_Hashtable == null)
				{
					m_Hashtable = new Hashtable();
					if (m_List != null)
						foreach (DBChange dbc in m_List)
							m_Hashtable.Add(dbc.ObjectId, dbc);
					m_List = null;
				}
					return m_Hashtable;
			}
		}

		public DBChange this[ObjectId id]
		{
			get
			{
				DBChange res = (DBChange)EnsuredHashTable[id];
				if (res == null)
				{
					res = new DBChange(id);
					Add(res);
				}
				return res;
			}
		}

		public DBChange GetEnsured(ObjectId id)
		{
			DBChange result = this[id];
			if (result == null)
			{
				result = new DBChange(id);
				EnsuredHashTable[id] = result;
			}
			return result;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="change"/> is null</exception>
		public void Add(DBChange change)
		{
			if (change == null)
				throw new ArgumentNullException("change"); // Do not localize
			EnsuredHashTable.Add(change.ObjectId, change);
		}

		public IEnumerator GetEnumerator()
		{
			return EnsuredHashTable.Values.GetEnumerator();
		}

		public int Count
		{
			get { return EnsuredHashTable.Count; }
		}

		public bool All
		{
			get	{ return m_All; }
			set { m_All = value; }
		}

	}
	[Serializable]
	public sealed class SyncVersion
	{
		private readonly int m_Version;
		private readonly Guid m_Guid;
		private static readonly SyncVersion m_None = new SyncVersion(-1, Guid.Empty);

		public SyncVersion(int version, Guid guid)
		{
			m_Version = version;
			m_Guid = guid;
		}

		public int Version
		{
			get { return m_Version; }
		}

		public Guid Guid
		{
			get { return m_Guid; }
		}

		public static SyncVersion None
		{
			get { return m_None; }
		}
	}

	public interface IPersistenceMapper
	{
		void Fetch(ObjectIdList idList, out Datablock datablock, int[] memberIdList);
		void FetchIDListWithCondition(AbstractCondition condition, out ObjectIdList result, int maxResults, int offset);
		void VersionForTime(DateTime clockTime, out int version);
		void TimeForVersion(int version, out DateTime clockTime);
		bool SupportsSync { get; }
		void Update(Datablock datablock, UpdatePrecondition precondition, out IdTranslationList translationList, out int version, out SyncVersion syncVersion, out UpdateResult result);
		void GetChangesSince(SyncVersion syncVersion, SyncVersion[] excludeList, out DBChangeCollection changes, out SyncVersion lastSyncVersion);
		SyncVersion CurrentSyncVersion { get; }
	}

	public interface IOptimisticLockHander
	{
		void AddOptimisticLocks(IEnumerable /* Locator*/ locators, OptimisticLockingPrecondition preCondition);
	}
	///<summary>
	///This interface is for internal use, and breaks the dependency between Borland.Eco.Handles.Windows and Borland.Eco.Persistence
	///(in the DBEvolution form and the ActionExtender).
	///</summary>
	public interface IPersistenceMapperDb
	{
		IDBEvolutor GetEvolutor(ITypeSystemService typeSystemService);
		void ReturnEvolutor(IDBEvolutor evolutor);
		void CreateDataBaseSchema(ITypeSystemService typeSystemService, IConfigureCleanPS configureCleanPS);
	}
}
